/***************************************************************************
 * launcher
 * Copyright (C) 2013 Blizzard Entertainment
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 ***************************************************************************/

#include "pathutils.h"
#include "utils.h"
#include "winutils.h"

#include <Shlwapi.h>


std::string app_path()
{
    WCHAR wpath[MAX_PATH];
    DWORD result;
    REQUIRE_NONZERO(result = ::GetModuleFileName(nullptr, wpath, lengthof(wpath)));
    if (result == lengthof(wpath))
        throw windows_error(ERROR_INSUFFICIENT_BUFFER);

    int wchar_count = lstrlen(wpath);
    int char_count = ::WideCharToMultiByte(CP_UTF8, 0, wpath, wchar_count, nullptr, 0, nullptr, nullptr);
    REQUIRE_NONZERO(char_count);

    std::string path(char_count, '\0');
    REQUIRE_NONZERO(::WideCharToMultiByte(CP_UTF8, 0, wpath, wchar_count, &path.front(), char_count, nullptr, nullptr));
    return path;
}

std::string& remove_last_path_component(std::string& path)
{
    ::PathRemoveBackslashA(&path.front());
    ::PathRemoveFileSpecA(&path.front());
    path.erase(path.find('\0'));
    return path;
}

int win_chdir(const char* path)
{
    if (path == nullptr)
    {
        errno = EINVAL;
        return -1;
    }

    WCHAR wpath[MAX_PATH];
    size_t path_length = strlen(path);
    int wpath_length = ::MultiByteToWideChar(CP_UTF8, 0, path, path_length, nullptr, 0);
    if (wpath_length == 0)
    {
        errno = EILSEQ;
        return -1;
    }

    if (wpath_length >= lengthof(wpath))
    {
        errno = ENAMETOOLONG;
        return -1;
    }

    ::MultiByteToWideChar(CP_UTF8, 0, path, path_length, wpath, wpath_length);
    wpath[wpath_length] = '\0';

    if (wpath[wpath_length - 1] != '/' && wpath[wpath_length - 1] != '\\')
    {
        if (wpath_length == lengthof(wpath) - 1)
        {
            errno = ENAMETOOLONG;
            return -1;
        }

        ::PathAddBackslash(wpath);
    }

    if (!::SetCurrentDirectory(wpath))
    {
        switch (::GetLastError())
        {
        case ERROR_FILE_NOT_FOUND:
        case ERROR_PATH_NOT_FOUND:
        case ERROR_INVALID_DRIVE:
            errno = ENOENT; break;
        case ERROR_ACCESS_DENIED:
            errno = EACCES; break;
        default:
            errno = EINVAL; break; // we don't know what you did, but it must have been your fault
        }

        return -1;
    }

    return 0;
}
